/**************************************************************************\
*//*! \file ef_bend_solarflare.c Solarflare specific code

Copyright 2006 Solarflare Communications Inc,
               9501 Jeronimo Road, Suite 250,
               Irvine, CA 92618, USA

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License version 2 as published by the Free
Software Foundation, incorporated herein by reference.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

 *//*
\**************************************************************************/

#include <linux/module.h>
#include <linux/netdevice.h>
#include "../../linux_net/efab/driverlink_api.h"
#include "ci/xen/ef_msg_iface.h"
#include "ef_bend.h"
#include "ef_char_bend.h"
#include "ef_bend_fwd.h"
#include "ef_bend_accel.h"


struct nic_desc {
  int type;
  char *name;
};

#define MAX_NICS (8)
struct nic_desc nics[MAX_NICS];
int num_nics;

/* Not all the MSG_HWTYPEs need to correspond to driverlink
 * EFAB_TYPEs, but for convenience we require that the ones that
 * do have the same value */
CI_BUILD_ASSERT(MSG_HWTYPE_EF1 == EFAB_TYPE_EF1002);
CI_BUILD_ASSERT(MSG_HWTYPE_FALCON == EFAB_TYPE_FALCON);

static struct efab_dl_device_id bend_dl_ids[] = {
#if defined(__CI_HARDWARE_CONFIG_EF1002__) 
{EFAB_TYPE_EF1002},
#endif
#if defined(__CI_HARDWARE_CONFIG_FALCON__) 
{EFAB_TYPE_FALCON},
#endif
{0}
};

static int init_done = 0;
/* Pointers to the driverlink functions, if present. */
int (*fn_efab_dl_register_driver) ( struct efab_dl_driver *driver );
void (*fn_efab_dl_unregister_driver) ( struct efab_dl_driver *driver );
int  (*fn_efab_dl_register_callbacks) ( struct efab_dl_device *efab_dev, struct efab_dl_callbacks *callbacks );
void (*fn_efab_dl_unregister_callbacks) ( struct efab_dl_device *efab_dev );

/* The DL callbacks */
/* Called on tx path, always allow */
/* The generic bend code doesn't know about the efab_evto enum, so check
 * everything is in sync. */
CI_BUILD_ASSERT(EFAB_ALLOW_PACKET == 0);
CI_BUILD_ASSERT(EFAB_VETO_PACKET == 1);
static  enum efab_veto fastcall bend_dl_tx_packet(struct efab_dl_port *efab_port, 
                                          struct sk_buff *skb )
{
  return EFAB_ALLOW_PACKET;
}

static enum efab_veto fastcall bend_dl_rx_packet(struct efab_dl_port *efab_port,
                                                  struct sk_buff *skb )
{
  return ef_bend_rx_packet(skb);
}

/* Called from netdriver when link state changes */
static void bend_dl_link_change( struct efab_dl_port *efab_port, int link_up )
{
  ci_log("Link state change for %s: %s", efab_port->net_dev->name, 
         link_up ? "up" : "down");
  ef_bend_link_change(link_up);
}


/* Called from netdriver when MTU change requested */
static int bend_dl_request_mtu( struct efab_dl_port *efab_port, int new_mtu )
{
  ci_log("MTU request to %d", new_mtu);
  /* No objections - doesn't make sense to check with each VM as what
     do we do if one objects and another thinks its fine? */
  return 0;
}


/* Called from netdriver when MTU changed */
static void bend_dl_mtu_changed( struct efab_dl_port *efab_port )
{
  int new_mtu = efab_port->net_dev->mtu;
  ef_bend_mtu_changed(new_mtu);
}


/* Callbacks we'd like to get from the netdriver through driverlink */
struct efab_dl_callbacks bend_dl_callbacks =
{
  .tx_packet = bend_dl_tx_packet,
  .rx_packet = bend_dl_rx_packet,
  .link_change = bend_dl_link_change,
  .request_mtu = bend_dl_request_mtu,
  .mtu_changed = bend_dl_mtu_changed
};

/* Driver link probe - register our callbacks */
static int bend_dl_probe ( struct efab_dl_device *efab_dev,
                           const struct efab_dl_device_id *entry )
{
  int rc;
  ci_log("probe called for a type %x device, name %s", entry->type, efab_dev->name);
  nics[num_nics].type = entry->type;
  nics[num_nics++].name = efab_dev->name;
  rc = fn_efab_dl_register_callbacks(efab_dev, &bend_dl_callbacks);
  ci_log("efab_dl_register_callbacks returned %d", rc);
  return rc;
}


static void bend_dl_remove ( struct efab_dl_device *efab_dev )
{
  ci_log("Unregistering driverlink callbacks.");
  fn_efab_dl_unregister_callbacks(efab_dev);
}

static  void bend_dl_reset_suspend( struct efab_dl_device *efab_dev )
{
}

static void bend_dl_reset_resume( struct efab_dl_device *efab_dev, int ok )
{
}

struct efab_dl_driver bend_dl_driver = 
{
  .name = "EF backend driver",
  .id_table = bend_dl_ids,
  .probe = bend_dl_probe,
  .remove = bend_dl_remove,
  .reset_suspend = bend_dl_reset_suspend,
  .reset_resume = bend_dl_reset_resume
};



  /* There are two separate sets of symbols we may or may not find:
  * those from the NET driver relating to driverlink and those from
  * the CHAR driver relating to resource management */
static int setup_driverlink(void)
{
  int rc = 0;
  fn_efab_dl_unregister_driver = symbol_get(efab_dl_unregister_driver);
  fn_efab_dl_register_driver = symbol_get(efab_dl_register_driver_api_ver_0104);
  fn_efab_dl_register_callbacks = symbol_get(efab_dl_register_callbacks);
  fn_efab_dl_unregister_callbacks = symbol_get(efab_dl_unregister_callbacks);
  if (fn_efab_dl_register_driver == NULL) {
    ci_log("Could not find NET driver symbols. Full slow path only.");
    rc = -ENOENT;
  } else {
    rc = fn_efab_dl_register_driver(&bend_dl_driver);
    ci_log("Driverlink returned %d", rc);
  }
  return rc;
}

extern int ef_bend_char_init(void);
extern void ef_bend_char_shutdown(void);

int ef_sf_accel_init(void)
{
  int rc = setup_driverlink();
  /* If we couldn't find the NET driver, we won't find the char driver, so
  * give up */
  if (rc == -ENOENT) {
    return rc;
  }
  rc = ef_bend_char_init();
  init_done = (rc == 0);
  return rc;
}

void ef_sf_accel_shutdown(void)
{
  if (!init_done)
    return;
  ci_log("Unregistering driverlink driver");
  /* This will trigger removal callbacks for all the devices, which
  * will unregister their callbacks */
  fn_efab_dl_unregister_driver(&bend_dl_driver);
  /* No-one must attempt to do anything hardware related after this as we
  * have just allowed the char driver to unload if it wants to */
  ef_bend_char_shutdown();
  symbol_put(efab_dl_unregister_driver);
  symbol_put(efab_dl_register_driver_api_ver_0104);
  symbol_put(efab_dl_register_callbacks);
  symbol_put(efab_dl_unregister_callbacks);
}

int ef_sf_accel_hwtype(struct ef_bend *bend)
{
  int rc = -ENOENT;
  int i;

  for (i = 0; i < num_nics; i++) {
    if (strcmp(bend->nicname, nics[i].name) == 0) {
      bend->hw_type = nics[i].type;
      bend->accel_setup = ef_bend_setup_vnic_hw;
      bend->accel_shutdown = ef_bend_shutdown_vnic_hw;
      rc = 0;
      break;
    }
  }
  if (rc != 0) {
    ci_log("Failed to identify backend device '%s' with a NIC", bend->nicname);
  }
  return rc;
}

